<?php
class Database {
    private static $instance = null;
    private $db;
    
    private function __construct() {
        $dbDir = dirname(DB_PATH);
        if (!file_exists($dbDir)) {
            mkdir($dbDir, 0755, true);
        }
        
        try {
            $this->db = new PDO('sqlite:' . DB_PATH);
            $this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->db->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
            $this->createTables();
        } catch (PDOException $e) {
            die('Database connection failed: ' . $e->getMessage());
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        return $this->db;
    }
    
    private function createTables() {
        $this->db->exec("
            CREATE TABLE IF NOT EXISTS products (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                csv_id INTEGER UNIQUE,
                name TEXT NOT NULL,
                description TEXT,
                price REAL NOT NULL,
                image_url TEXT,
                category TEXT,
                tags TEXT,
                keywords TEXT,
                popular INTEGER DEFAULT 0,
                permanentlink TEXT,
                paddle_product_id TEXT,
                github_release_tag TEXT,
                file_parts_count INTEGER DEFAULT 1,
                active INTEGER DEFAULT 1,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )
        ");
        
        $this->db->exec("
            CREATE TABLE IF NOT EXISTS orders (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                order_id TEXT UNIQUE NOT NULL,
                product_id INTEGER NOT NULL,
                customer_email TEXT NOT NULL,
                customer_name TEXT,
                amount REAL NOT NULL,
                currency TEXT DEFAULT 'USD',
                paddle_order_id TEXT,
                status TEXT DEFAULT 'pending',
                download_password TEXT,
                delivery_sent INTEGER DEFAULT 0,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (product_id) REFERENCES products(id)
            )
        ");
        
        $this->db->exec("
            CREATE TABLE IF NOT EXISTS coupons (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                code TEXT UNIQUE NOT NULL,
                discount_percent REAL NOT NULL,
                active INTEGER DEFAULT 1,
                max_uses INTEGER,
                current_uses INTEGER DEFAULT 0,
                expires_at DATETIME,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP
            )
        ");
        
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_products_csv_id ON products(csv_id)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_products_category ON products(category)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_products_popular ON products(popular)");
        $this->db->exec("CREATE INDEX IF NOT EXISTS idx_orders_order_id ON orders(order_id)");
    }
    
    public function importFromCSV($csvPath) {
        if (!file_exists($csvPath)) {
            return ['success' => false, 'message' => 'CSV file not found'];
        }
        
        $handle = fopen($csvPath, 'r');
        if (!$handle) {
            return ['success' => false, 'message' => 'Cannot open CSV file'];
        }
        
        fgetcsv($handle, 0, '|');
        
        $count = 0;
        $updated = 0;
        $skipped = 0;
        
        $stmt = $this->db->prepare("
            INSERT OR REPLACE INTO products 
            (csv_id, name, description, price, image_url, category, tags, keywords, popular, permanentlink, 
             paddle_product_id, github_release_tag, file_parts_count, active)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1, 1)
        ");
        
        while (($data = fgetcsv($handle, 0, '|')) !== false) {
            if (count($data) < 11) {
                $skipped++;
                continue;
            }
            
            try {
                $csvId = intval($data[0]);
                $title = $data[1];
                $description = $data[2];
                $price = floatval($data[3]);
                $image = $data[4];
                $category = $data[5];
                $tags = $data[6];
                $keywords = $data[7];
                $popular = strtolower($data[8]) === 'true' ? 1 : 0;
                $permanentlink = $data[9];
                
                $stmt->execute([
                    $csvId,
                    $title,
                    $description,
                    $price,
                    $image,
                    $category,
                    $tags,
                    $keywords,
                    $popular,
                    $permanentlink,
                    "prod_" . $csvId,
                    "product-" . $csvId
                ]);
                
                $count++;
            } catch (Exception $e) {
                $skipped++;
            }
        }
        
        fclose($handle);
        
        return [
            'success' => true,
            'message' => "Import completed: $count products processed, $skipped skipped"
        ];
    }
}
?>
